#!/usr/bin/env python3

import os

# Which coin represents fiat in your trades (ie. your currency for tax purposes)
FIAT = "GBP"
SYMBOL = "£"

# Coins that you have traded in the period
coins = [ "USDC", "GBP", 
    "ALGO", "BTC", "CHZ", "COMP", "CRV", "ENJ", "EOS", "ETH", "LINK", "LTC", "MATIC", "XLM", "XRP", "XTZ",
    "OMG", "AAVE"
]

wal = {}
sells = []

for fname in os.listdir("."):
    if fname.endswith(".csv"):
        
        with open(fname, "r") as f:
            data = f.readlines()
        
        for l in data:
            timestamp, tid, oid, product, side, liquidity, price, size = l.split(",")
            if timestamp.startswith("Time"): continue # skip header

            total = float(price) * float(size)
            lcoin = ""
            rcoin = ""
            for c in coins:
                if product.startswith(c): lcoin = c
                if product.endswith(c): rcoin = c

            if lcoin == "" or rcoin == "": print("MISSING COIN: %s" % product)

            d = { "avg": 0.0, "coins": 0.0, "buy": 0.0, "sell": 0.0, "lastprice": 0.0, "numtrx": 0, "totprc": 0.0 }
            if lcoin in wal:
                d = wal[lcoin]
            else:
                wal[lcoin] = d

            is_fiat = rcoin == FIAT

            if is_fiat:
                d["lastprice"] = float(price)
            else:
                # lastprice should always be in fiat - if it isn't, find the fiat price
                # for the traded coin and use that
                if rcoin in wal: d["lastprice"] = wal[rcoin]["lastprice"] * float(price)

            if side == "buy":
                d["numtrx"] += 1
                d["coins"] += float(size)
                d["totprc"] += d["lastprice"]
                if is_fiat:
                    d["buy"] += float(total)
                    d["lastbuy"] = float(price)
                else:
                    lastprice = 0.0
                    if rcoin in wal: lastprice = wal[rcoin]["lastprice"]
                    if lastprice > 0:
                        d["lastbuy"] += lastprice # convert to fiat
                        d["buy"] += ( float(total) * lastprice) # convert to fiat
            else:
                d["coins"] -= float(size)
                if is_fiat:
                    d["sell"] += float(total)
                    # Construct a fiat sell object for our close position
                    if "lastbuy" not in wal[lcoin]:
                        lastbuy = float(price) # we don't have a previous buy in the files so we're just going to have to use the sell price as the cost basis
                    else:
                        lastbuy = wal[lcoin]["lastbuy"]
                    buyfiat = float(lastbuy) * float(size)
                    sells.append({ 
                        "date": timestamp,
                        "coin": lcoin, 
                        "size": float(size), 
                        "sellprice": float(price), 
                        "buyprice": float(lastbuy), 
                        "buyfiat": buyfiat,
                        "sellfiat": float(total), 
                        "profitgbp": float(total) - buyfiat
                    })
                else:
                    if rcoin in wal: d["sell"] += float(total) * wal[rcoin]["lastprice"]

for k,v in wal.items():
    # Calculate average
    if v["numtrx"] > 0: v["avg"] = v["totprc"] / v["numtrx"]

print("Holding:")
print("")

for k in sorted(wal.keys()):
    v = wal[k]
    s = "%s: holding %0.4f (£%0.2f), avg price %0.4f, last price %0.4f, buy £%0.2f, sell £%0.2f" % ( k, v["coins"], v["coins"] * v["lastprice"], v["avg"], v["lastprice"], v["buy"], v["sell"])
    print(s.replace("£", SYMBOL))

print("")
print("Closed positions:")
print("")
totalprofit = 0.0
for p in sells:
    s = "%s, %0.4f %s. buy price £%0.4f, sell price £%0.4f, cost basis £%0.2f, sold for £%0.2f, profit £%0.2f" % \
        (p["date"], p["size"], p["coin"], p["buyprice"], p["sellprice"], p["buyfiat"], p["sellfiat"], p["profitgbp"])
    print(s.replace("£", SYMBOL))
    totalprofit += p["profitgbp"]
print("")
s = "Total profit: £%0.2f" % totalprofit
print(s.replace("£", SYMBOL))

