/*
 
 in.www.c
 Copyright(c)2008, R. Rawson-Tetley

 Simple, static HTTP server for use with inetd and my
 NSLU2 (slug) - I use this for serving mp3s and CGI
 handling.

 Supports static content, GET requests and basic CGI.


 This program is free software; you can redistribute it and/or
 modify it under the terms of the GNU General Public License as
 published by the Free Software Foundation; either version 2 of
 the License, or (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTIBILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the
 Free Software Foundation, Inc., 59 Temple Place - Suite 330, Boston
 MA 02111-1307, USA.

 Contact me by electronic mail: bobintetley@users.sourceforge.net
 */

#include <stdio.h>
#include <string.h>
#include <sys/stat.h>
#include <dirent.h>
#include <time.h>

#define SERVER "in.www/1.01"
#define DEFAULT_MIMETYPE "text/plain"
#define PROTOCOL "HTTP/1.0"
#define RFC1123FMT "%a, %d %b %Y %H:%M:%S GMT"

// Output buffer size for CGI calls
#define CGI_BUFFER 262140
// Buffer size for each CGI environment variable and input buffer
#define TOKEN_SIZE 4096

char* get_mime_type(char* name)
{
    // This is really crude, but quick. Could probably use
    // libmagic for accuracy
    char* ext = strrchr(name, '.');
    if (!ext) return DEFAULT_MIMETYPE;
    if (strcmp(ext, ".html") == 0 || strcmp(ext, ".htm") == 0) return "text/html";
    if (strcmp(ext, ".jpg") == 0 || strcmp(ext, ".jpeg") == 0) return "image/jpeg";
    if (strcmp(ext, ".txt") == 0) return "text/plain";
    if (strcmp(ext, ".csv") == 0) return "text/csv";
    if (strcmp(ext, ".gif") == 0) return "image/gif";
    if (strcmp(ext, ".png") == 0) return "image/png";
    if (strcmp(ext, ".css") == 0) return "text/css";
    if (strcmp(ext, ".au") == 0) return "audio/basic";
    if (strcmp(ext, ".wav") == 0) return "audio/wav";
    if (strcmp(ext, ".avi") == 0) return "video/x-msvideo";
    if (strcmp(ext, ".mpeg") == 0 || strcmp(ext, ".mpg") == 0) return "video/mpeg";
    if (strcmp(ext, ".mp3") == 0) return "audio/mpeg";
    
    #ifdef CGI
    if (strcmp(ext, ".cgi") == 0) return "cgi";
    #else
    if (strcmp(ext, ".cgi") == 0) return "text/plain";
    #endif

    return DEFAULT_MIMETYPE;
}

void send_headers(int status, char* title, char* extra, char* mime, 
        int length, time_t date, int complete)
{
    time_t now;
    char timebuf[128];

    printf("%s %d %s\r\n", PROTOCOL, status, title);
    printf("Server: %s\r\n", SERVER);
    now = time(NULL);
    strftime(timebuf, sizeof(timebuf), RFC1123FMT, gmtime(&now));
    printf("Date: %s\r\n", timebuf);
    if (extra) printf("%s\r\n", extra);
    if (mime) printf("Content-Type: %s\r\n", mime);
    if (length >= 0) printf("Content-Length: %d\r\n", length);
    if (date != -1)
    {
        strftime(timebuf, sizeof(timebuf), RFC1123FMT, gmtime(&date));
        printf("Last-Modified: %s\r\n", timebuf);
    }
    if (complete > 0) 
    {
        printf("Connection: close\r\n");
        printf("\r\n");
    }
}

void send_error(int status, char* title, char* extra, char* text)
{
    send_headers(status, title, extra, "text/html", -1, -1, 1);
    printf("<HTML><HEAD><TITLE>%d %s</TITLE></HEAD>\r\n", status, title);
    printf("<BODY><H4>%d %s</H4>\r\n", status, title);
    printf("%s\r\n", text);
    printf("</BODY></HTML>\r\n");
}

#ifdef CGI
void handle_cgi(char* path, char* vpath, char* querystring)
{
    // CGI output buffer
    char output[CGI_BUFFER];

    // Set the environment variables for the CGI
    char server_software[TOKEN_SIZE];
    snprintf(server_software, sizeof(server_software), "SERVER_SOFTWARE=%s", SERVER);

    char server_name[TOKEN_SIZE];
    char* hostname = (char*) getenv("HOSTNAME");
    if (!hostname) hostname = "";
    snprintf(server_name, sizeof(server_name), "SERVER_NAME=%s", hostname);

    char script_name[TOKEN_SIZE];
    char* qs = strstr(vpath, "?");
    if (qs) *(qs) = '\0';
    snprintf(script_name, sizeof(script_name), "SCRIPT_NAME=%s", vpath);

    char query_string[TOKEN_SIZE];
    snprintf(query_string, sizeof(query_string), "QUERY_STRING=%s", querystring);

    putenv(server_software);
    putenv(server_name);
    putenv(script_name);
    putenv(query_string);
    putenv("GATEWAY_INTERFACE=CGI/1.0");
    putenv("SERVER_PROTOCOL=HTTP/1.0");
    putenv("REQUEST_METHOD=GET");
    putenv("PATH_INFO=");
    putenv("PATH_TRANSLATED=");

    // Execute the CGI and grab the output
    FILE *p = popen(path, "r");
    int bytesread = fread(output, 1, sizeof(output), p);
    int retval = pclose(p);

    // Send content or error based on return code
    if (retval == 0) 
    {
        send_headers( 200, "OK", NULL, NULL, bytesread, -1, 0 );
        fwrite(output, 1, bytesread, stdout);
    }
    else 
    {
        send_error( 500, "Internal Server Error", NULL, "An error occurred serving the document requested." ); 
    }
}
#endif // CGI

void send_file(char* path, struct stat *statbuf, char* vpath, char* querystring)
{
    char data[4096];
    int n;
    char* mime_type = get_mime_type(path);

    FILE *file = fopen(path, "r");
    if (!file) 
    {
        send_error(403, "Forbidden", NULL, "Access denied.");
        return;
    }

    #ifdef CGI
    // Handle as a CGI script if that's what it is
    if (strcmp(mime_type, "cgi") == 0) 
    {
        fclose(file);
        handle_cgi(path, vpath, querystring);
        return;
    }
    #endif

    // Send the file
    int length = S_ISREG(statbuf->st_mode) ? statbuf->st_size : -1;
    send_headers( 200, "OK", NULL, mime_type, length, statbuf->st_mtime, 1);

    while ((n = fread(data, 1, sizeof(data), file)) > 0) fwrite(data, 1, n, stdout);
    fclose(file);
}

int process(char* documentbase)
{
    char buf[TOKEN_SIZE];
    char* method;
    char* path;
    char* protocol;
    char* querystring;
    struct stat statbuf;
    char pathbuf[TOKEN_SIZE];
    char realpath[TOKEN_SIZE];
    int len;

    if (!fgets(buf, sizeof(buf), stdin)) return -1;

    // Get the individual bits of the request
    method = strtok(buf, " ");
    path = strtok(NULL, " ");
    protocol = strtok(NULL, "\r");

    // Stop if the request is incomplete
    if (!method || !path || !protocol) return -1;

    // Parse querystring if there is one
    querystring = strstr(path, "?");
    if (querystring)
        querystring += sizeof(char);

    // Use the documentbase to find the path to the document
    snprintf(realpath, sizeof(realpath), "%s%s", documentbase, path);

    // Throw away the querystring if there is one
    char* qsloc = strstr(realpath, "?");
    if (qsloc)
        *(qsloc) = '\0';

    // We only support GET requests
    if (strcmp(method, "GET") != 0)
        send_error(501, "Not supported", NULL, "Method is not supported.");

    #ifdef NORELATIVE
    // Send forbidden for relative paths if we're stripping
    else if (strstr(realpath, ".."))
        send_error(403, "Forbidden", NULL, "Permission denied.");
    #endif

    // Drop out if the path is invalid
    else if (stat(realpath, &statbuf) < 0)
        send_error(404, "Not Found", NULL, "File not found.");

    // Send directory listing
    else if (S_ISDIR(statbuf.st_mode))
    {
        len = strlen(realpath);
        if (len == 0 || realpath[len - 1] != '/')
        {
            snprintf(pathbuf, sizeof(pathbuf), "Location: %s/", path);
            send_error(302, "Found", pathbuf, "Directories must end with a slash.");
        }
        else
        {
            snprintf(pathbuf, sizeof(pathbuf), "%s%sindex.html", documentbase, path);
            if (stat(pathbuf, &statbuf) >= 0)
                send_file(pathbuf, &statbuf, NULL, NULL);
            else
            {
                DIR *dir;
                struct dirent *de;

                send_headers(200, "OK", NULL, "text/html", -1, statbuf.st_mtime, 1);
                printf("<HTML><HEAD><TITLE>Index of %s</TITLE></HEAD>\r\n<BODY>", path);
                printf("<H4>Index of %s</H4>\r\n<PRE>\n", path);
                printf("Name Last Modified Size\r\n");
                printf("<HR>\r\n");
                if (len > 1) printf("<A HREF=\"..\">..</A>\r\n");

                dir = opendir(realpath);
                while ((de = readdir(dir)) != NULL)
                {
                    char timebuf[32];
                    struct tm *tm;

                    strcpy(pathbuf, realpath);
                    strcat(pathbuf, de->d_name);

                    stat(pathbuf, &statbuf);
                    tm = gmtime(&statbuf.st_mtime);
                    strftime(timebuf, sizeof(timebuf), "%d-%b-%Y %H:%M:%S", tm);

                    printf("<A HREF=\"%s%s\">", de->d_name, S_ISDIR(statbuf.st_mode) ? "/" : "");
                    printf("%s%s", de->d_name, S_ISDIR(statbuf.st_mode) ? "/</A>" : "</A> ");
                    if (strlen(de->d_name) < 32) printf("%*s", 32 - strlen(de->d_name), "");

                    if (S_ISDIR(statbuf.st_mode))
                        printf("%s\r\n", timebuf);
                    else
                        printf("%s %10d\r\n", timebuf, statbuf.st_size);
                }
                closedir(dir);

                printf("</PRE>\r\n<HR>\r\n<ADDRESS>%s</ADDRESS>\r\n</BODY></HTML>\r\n", SERVER);
            }
        }
    }
    else
    {
        send_file(realpath, &statbuf, path, querystring);
    }

    return 0;
}

int main(int argc, char* argv[])
{
    char* docbase = "/var/www";
    if (argc > 1)
        docbase = argv[1];
    process(docbase);
    return 0;
}
