package net.sf.umlspeed.parser;

import java.util.List;

import net.sf.umlspeed.cli.CLI;
import net.sf.umlspeed.entities.DataStore;

/**
 * Base class of all parsers (classes that syntax check and parse a
 * particular token/enclosure from the buffer).
 * As well as being the base class, this is also the top-level Parser
 * that creates the child parsers for each type of entity (see parse method).
 */
public class Parser extends Lexer {
    
    public Parser() { }
    
    public Parser(String filename, String buffer, int currentPosition) {
        super(filename, buffer, currentPosition);
    }
    
    protected void checkEntityNameIsFree(String name) {
        if (DataStore.entities.get(name) != null) {
            parseError("Duplicate entity name '" + name + "'");
        }
    }
    
    /** Looks in findin for all occurrences of find and replaces them with replacewith 
     * @param findin The string to find occurrences in
     * @param find The string to find
     * @param replacewith The string to replace found occurrences with
     * @return A string with all occurrences of find replaced.
     */
    protected String replace(String findin, String find, String replacewith) {
        
        StringBuffer sb = new StringBuffer(findin);
        int i = 0;
        try {
            while (i <= sb.length() - find.length()) {
                if (sb.substring(i, i + find.length()).equalsIgnoreCase(find)) {
                    sb.replace(i, i + find.length(), replacewith);
                }
                i++;
            }
        }
        catch (StringIndexOutOfBoundsException e) {
            // We hit the end of the string - do nothing and carry on
        }
        return sb.toString();
    }
    
    /**
     * Splits a string by a particular char and returns an array of 
     * strings. If there are no occurrences of the split char, the
     * original string is returned in an array of 1 item.
     * @param splitstring The string to be split
     * @param splitchar The char to split on
     * @return An array of strings
     */
    protected String[] split(String splitstring, String splitchar) {
       
        splitstring = splitstring.trim();
        
        // If there is only one element, just return that
        if (splitstring.indexOf(splitchar) == -1) {
            String[] rets = new String[1];
            rets[0] = splitstring;
            return rets;
        }
        
        // Find how many there are
        int tot = 0;
        int lpos = splitstring.indexOf(splitchar);
        while (lpos != -1) {
            tot++;
            lpos = splitstring.indexOf(splitchar, lpos + 1);
        }
        tot++;
        
        // Create our new array
        String[] rets = new String[tot];
        tot = 0;
        lpos = 0;
        int spos = splitstring.indexOf(splitchar);
        while (spos != -1) {
            // Add into the array
            rets[tot] = splitstring.substring(lpos, spos);
            tot++;
            lpos = spos + 1;
            spos = splitstring.indexOf(splitchar, lpos);
        }
        
        // Include last word
        rets[tot] = splitstring.substring(lpos, splitstring.length());
        
        // Return it
        return rets;
    }
    
    protected void arrayToList(String[] arr, List l) {
        for (int i=0; i < arr.length; i++)
            l.add(arr[i].trim());
    }
    
    protected String removeQuotes(String s) {
        if (s.startsWith("\""))
            s = s.substring(1);
        if (s.endsWith("\""))
            s = s.substring(0, s.length()-1);
        return s;
    }
    
    protected String removeBrackets(String s) {
        if (s.startsWith("("))
            s = s.substring(1);
        if (s.endsWith(")"))
            s = s.substring(0, s.length()-1);
        return s;
    }
    
    protected void parseError(String message) {
        
        // Find the line number where the error occurred and
        // the start of the line with the error on it.
        int lineno = 1;
        int lastlinestart = 0;
        for (int i = 0; i < currentPosition; i++) {
            if (buffer.substring(i, i+1).equals("\n")) {
                lineno++;
                lastlinestart = i + 1;
            }
        }
        
        // Output the message and line number/col
        CLI.print("Syntax Error in '" + filename + "' at Line " + lineno + ", Column " + (currentPosition - lastlinestart));
        CLI.print(message);
        
        // Show line of code where error occurred
        int endpoint = buffer.indexOf("\n", currentPosition);
        if (endpoint == -1) endpoint = buffer.length();
        CLI.print("   " + buffer.substring(lastlinestart, endpoint));
        
        System.exit(1);
        
    }
    
    protected void assertToken(String s) {
        if (!currentToken.equals(s)) parseError(s + " expected.");
    }
    
    /**
     * Does the lexing/parsing work. Since this is the top-level base
     * method, the return value of buffer position is irrelevant as we
     * have completed lexing/parsing of the whole buffer on return from here.
     * @return The new buffer position.
     */
    public int parse() {
        
        while (true) {
        
            // Find next available token
            findNextToken();
            if (currentPosition == -1) break;
            
            // Evaluate and handle the token
            if (currentToken.equals("import")) {
                currentPosition = new ImportParser(filename, buffer, currentPosition).parse();
            }
            else if (currentToken.equals("class")) {
                currentPosition = new ClassParser(filename, buffer, currentPosition).parse();
            }
            else if (currentToken.equals("interface")) {
                currentPosition = new InterfaceParser(filename, buffer, currentPosition).parse();
            }
            else if (currentToken.equals("classdiagram")) {
                currentPosition = new ClassDiagramParser(filename, buffer, currentPosition).parse();
            }
            else if (currentToken.equals("actor")) {
                currentPosition = new ActorParser(filename, buffer, currentPosition).parse();
            }
            else if (currentToken.equals("usecase")) {
                currentPosition = new UseCaseParser(filename, buffer, currentPosition).parse();
            }
            else if (currentToken.equals("usecasediagram")) {
                currentPosition = new UseCaseDiagramParser(filename, buffer, currentPosition).parse();
            }
            else if (currentToken.equals("namespace")) {
                currentPosition = new NamespaceParser(filename, buffer, currentPosition).parse();
            }
            else {
                parseError("Unrecognised token: " + currentToken);
            }
        }
        return -1;
    }
}
