package net.sf.umlspeed.parser;

import net.sf.umlspeed.cli.CLI;
import net.sf.umlspeed.entities.DataStore;
import net.sf.umlspeed.entities.DiagramElement;
import net.sf.umlspeed.entities.DiagramLink;
import net.sf.umlspeed.entities.Entity;
import net.sf.umlspeed.entities.UseCaseDiagram;

/**
 * Handles lexing/parsing of usecase diagrams, eg:
 * 
 *      usecasediagram mydiagram {
 *          comment = "My diagram";
 *          entities {
 *              thing1 uses thing2;
 *              thing2 extends thing3;
 *              thing3 includes thing2;
 *          }
 *      }
 *      
 * Spawns a new Lexer, buffers in the file and parses its entities
 * into our set.
 */
public class UseCaseDiagramParser extends Parser {
    
    private UseCaseDiagram dg = new UseCaseDiagram();
    
    public UseCaseDiagramParser(String filename, String buffer, int currentPosition) {
        super(filename, buffer, currentPosition);
    }
    
    /**
     * Does the lexing/parsing work. Returns the new position in the buffer
     * after all lexing/parsing has been done for this token.
     * @return The new buffer position.
     */
    public int parse() {
        
        // Get next argument. This should be the diagram name
        findNextToken();
        dg.setName(currentToken);
        CLI.print("UseCaseDiagramParser: " + dg.getName(), 1);
        checkEntityNameIsFree(currentToken);
        
        // Get next argument, this should be the opening brace to start
        // the enclosure
        findNextToken();
        assertToken("{");
        
        // Loop round looking for inner diagram enclosures - comment and entities:
        while (true) {
            
            findNextToken();
            if (currentToken.equals("comment")) {
                parseComment();
            }
            else if (currentToken.equals("entities")) {
                parseEntities();
            }
            else if (currentToken.equals("layout")) {
                parseLayout();
            }
            else if (currentToken.equals("}")) {
                // That's the end of the entities enclosure, break out and finish.
                CLI.print("UseCaseDiagramParser: end of diagram.", 2);
                DataStore.diagrams.put(dg.getName(), dg);
                DataStore.entities.put(dg.getName(), dg);
                break;
            }
            else {
                parseError("'" + currentToken + "' is not valid here, expected comment, layout or entities.");
            }
            
        }
        return currentPosition;
    }
    
    /**
     * Searches the diagram elements, looking for the entity named.
     * If the entity doesn't exist in the diagram elements, we create
     * it and return the new object back to the caller.
     * Working this way, we build up a map of unique objects that
     * only appear once on the diagram and can recursively link to
     * each other.
     * 
     * @param name The name of the entity
     * @param entity The entity object
     */
    private DiagramElement findElement(String name, Entity entity) {
        DiagramElement el = (DiagramElement) dg.getElements().get(name);
        if (el != null) CLI.print("ClassDiagramParser: findElement: Got existing '" + name + "' entity.", 3);
        if (el == null) {
            el = new DiagramElement();
            el.setEntityName(name);
            el.setEntity(entity);
            dg.getElements().put(name, el);
            CLI.print("ClassDiagramParser: findElement: Created new '" + name + "' entity.", 3);
        }
        return el;
    }
    
    /** Parse the comment enclosure */
    private void parseComment() {
        findNextToken();
        assertToken("=");
        findNextToken();
        dg.setComment(removeQuotes(currentToken));
        findNextToken();
        assertToken(";");
    }
    
    private void parseLayout() {
        
        findNextToken();
        assertToken("=");
        
        findNextToken();
        dg.setLayout(removeQuotes(currentToken));
        
        // Verify we have a valid layout
        if (!currentToken.equals("satellite") &&
            !currentToken.equals("usecase") &&
            !currentToken.equals("grid"))
            parseError("'" + currentToken + "' is not a valid diagram layout.");
        CLI.print("UseCaseDiagramParser: Using " + currentToken + " layout", 2);
        
        findNextToken();
        
        // Should have a bracketed list of arguments or a terminator
        if (!currentToken.startsWith("(") && !currentToken.equals(";"))
            parseError("Expected layout argument list or ;");
        
        if (currentToken.startsWith("(")) {
            dg.setLayoutArgs(split(removeBrackets(currentToken), ","));
            for (int i = 0; i < dg.getLayoutArgs().length; i++)
                dg.getLayoutArgs()[i] = dg.getLayoutArgs()[i].replace('"', ' ').trim();
            
            CLI.print("UseCaseDiagramParser: Layout arguments " + currentToken, 2);
            
            findNextToken();
            assertToken(";");
        }
        else {
            // If the layout manager is satellite or grid and we got here,
            // no argument list was given.
            if (dg.getLayout().equals("grid") || dg.getLayout().equals("satellite"))
                parseError("satellite and grid layout require layout arguments.");
        }
    }
    
    private void parseEntities() {
        
        findNextToken();
        assertToken("{");
        
        while (true) {
            
            findNextToken();
            if (currentToken.equals("}"))
                // End of the enclosure
                break;
            
            // Should be an entity name
            Entity e = (Entity) DataStore.entities.get(currentToken);
            if (e == null)
                parseError("Entity '" + currentToken + "' does not exist, or has not been parsed yet.\nPut your diagram below the entity declaration.");
            
            // Either find an existing element if we have one, or create one
            DiagramElement el = findElement(currentToken, e);
            CLI.print("UseCaseDiagramParser: left join entity " + currentToken, 2);

            // Relationship and second entity or terminator
            findNextToken();
            if (!currentToken.equals(";")) {
                
                // Create a link object
                DiagramLink dl = new DiagramLink();
                if (currentToken.equals("extends"))
                    dl.setLinkType(DiagramLink.LINK_EXTENDS);
                else if (currentToken.equals("includes"))
                    dl.setLinkType(DiagramLink.LINK_INCLUDES);
                else if (currentToken.equals("uses"))
                    dl.setLinkType(DiagramLink.LINK_USES);
                else {
                    parseError("Invalid relationship '" + currentToken + "'");
                }
                CLI.print("UseCaseDiagramParser: " + el.getEntityName() + " " + currentToken, 2);
                
                // Target entity
                findNextToken();
                e = (Entity) DataStore.entities.get(currentToken);
                if (e == null)
                    parseError("Entity '" + currentToken + "' does not exist, or has not been parsed yet.\nPut your diagram below the entity declaration.");
                CLI.print("UseCaseDiagramParser: right join entity " + currentToken, 2);
                
                // Find it in our existing diagram elements if we have it, or
                // create it if not
                DiagramElement el2 = findElement(currentToken, e);
                
                // Make the link from the first element to the second one
                dl.setTargetEntity(el2);
                
                // Assign it to the first element
                el.getLinks().add(dl);
                CLI.print("UseCaseDiagramParser: Link Creation (" + el.getLinks().size() + ")", 2);
                
                // We should have a terminator next, ready for the next element
                findNextToken();
                assertToken(";");
            }
            
        }
        
    }
    
}
