package net.sf.umlspeed.svg;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import net.sf.umlspeed.cli.CLI;
import net.sf.umlspeed.entities.Actor;
import net.sf.umlspeed.entities.DiagramElement;

/**
 * Manages the layout of SVG objects in a grid of
 * X columns, with the entities specified in order
 * by the layout arguments.
 */
public class SVGGridLayout extends SVGLayout {

    private Map objects = new HashMap();
    private int noColumns = 0;
    private String[] layoutArgs = null;
    
    private int canvaswidth = 0;
    private int canvasheight = 0;

    public SVGGridLayout(int noColumns, String[] layoutArgs) {
        this.noColumns = noColumns;
        this.layoutArgs = layoutArgs;
    }
    
    public void add(SVGEntity e) {
        objects.put(e.getEntity().getName(), e);
    }

    public void remove(SVGEntity e) {
        objects.remove(e);
    }
    
    public int getWidth() {
        return canvaswidth;
    }
    
    public int getHeight() {
        return canvasheight;
    }

    public List getObjects() {
        ArrayList l = new ArrayList();
        for (Iterator it = objects.values().iterator(); it.hasNext(); )
            l.add(it.next());
        return l;
    }
    
    public String render() {
        
        StringBuffer svg = new StringBuffer();
        
        // We arrange the objects into noColumns columns and as many
        // rows as we need to display the layout args
        int noRows = (layoutArgs.length-1) / noColumns;
        
        // Make a quick pass through the objects to figure out
        // the largest item we're dealing with.
        int biggestwidth = 0;
        int biggestheight = 0;
        for (Iterator it = objects.values().iterator(); it.hasNext(); ) {
            SVGEntity e = (SVGEntity) it.next();
            e.setPosition(0, 0);
            if (e.getSize().width > biggestwidth)
                biggestwidth = e.getSize().width;
            if (e.getSize().height > biggestheight)
                biggestheight = e.getSize().height;
        }
        
        // The canvasmargin is the gap we allow between objects and the
        // edge of the canvas
        int canvasmargin = 20;
        
        // Margin is the gap (horizontal and vertical) between the nearest
        // diagram element. The fewer the elements, the larger the margin
        // should be for clarity.
        int margin = 10;
        if ( objects.size() < 3)
            margin = 100;
        else if ( objects.size() < 5)
            margin = 70;
        else if ( objects.size() < 10)
            margin = 50;
        else if ( objects.size() < 20)
            margin = 20;
        else
            margin = 10;
        
        // Render the objects
        int x = canvasmargin;
        int y = canvasmargin;
        int curcol = 1;
        for (int i = 1; i < layoutArgs.length; i++) {
            
            if (!layoutArgs[i].equals("*")) {
                SVGEntity e = (SVGEntity) objects.get(layoutArgs[i]);
                
                // If we have no entity, someone's made a booboo and not included
                // it in the entities list.
                if (e == null) {
                    CLI.print("SVGGridLayout: Layout specifies entity '" + 
                         layoutArgs[i] + "' but it doesn't exist in the entities list.");
                    System.exit(1);
                }
                e.setPosition(centerInCell(new SVGDimension(biggestwidth, biggestheight), e.getSize(), new SVGPosition(x, y)));
                svg.append(e.getSVG());
            }
            
            curcol++;
            if (curcol > noColumns) {
                curcol = 1;
                x = canvasmargin;
                y += margin + biggestheight;
            }
            else {
                x += margin + biggestwidth;
            }
        }
        
        // Set overall width/height (might want them later)
        canvaswidth = (canvasmargin * 2) + (noColumns * (biggestwidth + margin));
        canvasheight = (canvasmargin * 2) + (noRows * (biggestheight + margin));
        
        return svg.toString();
    }

}
