package net.sf.umlspeed.svg;

import java.io.File;
import java.io.FileOutputStream;
import java.util.Iterator;
import java.util.List;

import net.sf.umlspeed.cli.CLI;
import net.sf.umlspeed.entities.Actor;
import net.sf.umlspeed.entities.Class;
import net.sf.umlspeed.entities.ClassDiagram;
import net.sf.umlspeed.entities.DataStore;
import net.sf.umlspeed.entities.Diagram;
import net.sf.umlspeed.entities.DiagramElement;
import net.sf.umlspeed.entities.DiagramLink;
import net.sf.umlspeed.entities.Entity;
import net.sf.umlspeed.entities.Interface;
import net.sf.umlspeed.entities.UseCase;
import net.sf.umlspeed.entities.UseCaseDiagram;

/**
 * Responsible for rendering entities into an SVG document.
 */
public class SVGRenderer {

    private SVGLayout layout = null;
    private StringBuffer svg = new StringBuffer();
    private int height = 0;
    private int width = 0;
    private int canvasmargin = 20;
    
    /** 
     * Renders all diagram entities as SVG documents
     */
    public void renderAllDiagrams() {
        CLI.print("SVGRenderer: Rendering all diagrams", 2);
        boolean renderedOne = false;
        for (Iterator it = DataStore.diagrams.keySet().iterator(); it.hasNext(); ) {
            render(it.next().toString());
            svg = new StringBuffer();
            renderedOne = true;
        }
        if (!renderedOne) {
            CLI.print("SVGRenderer: SVG Output specified, but no diagrams in input file.");
            System.exit(1);
        }
    }
    
    public void renderError(String message) {
        CLI.print("ERROR: === " + message);
        System.exit(1);
    }
    
    /** SVG Document Header */
    private String renderHeader(int width, int height) {
        this.height = height + canvasmargin;
        this.width = width;
        StringBuffer s = new StringBuffer();
        s.append("<?xml version=\"1.0\" standalone=\"no\"?>\n");
        s.append("<!DOCTYPE svg PUBLIC \"-//W3C//DTD SVG 1.0//EN\" \n");
        s.append("\"http://www.w3.org/TR/2001/REC-SVG-20010904/DTD/svg11.dtd\" >\n");
        s.append("<svg version=\"1.1\" xmlns=\"http://www.w3.org/2000/svg\" ");
        s.append("width=\"" + width + "\" height=\"" + height + "\"");
        s.append(">\n\n");
        s.append("<!-- Generated by UMLSpeed " + CLI.VERSION + " -->\n\n");
        return s.toString();
    }
    
    /** SVG Document Footer */
    private String renderFooter() {
        return "</svg>";
    }
    
    /** 
     * Renders the given entity into a single SVG document.
     * This is generally for use with diagram descendants, but if another 
     * entity type is passed, the single entity will be rendered.
     */
    public void render(String entityName) {
        
        CLI.print("SVGRenderer: Rendering " + entityName, 1);
        
        Entity e = (Entity) DataStore.entities.get(entityName);
        if (e == null)
            renderError("Entity '" + entityName + "' does not exist.");
        
        // What kind of entity are we dealing with?
        if (e instanceof ClassDiagram)
            renderClassDiagram((ClassDiagram) e);
        else if (e instanceof UseCaseDiagram)
            renderUseCaseDiagram((UseCaseDiagram) e);
        else
            // It's just an entity of some type
            renderSingleEntity(e);
        
        // Render the diagram figure at the bottom
        if (e instanceof Diagram) {
            SVGTextRenderer se = new SVGTextRenderer(((Diagram) e).getComment(), SVGEntity.MEDIUM_HEIGHT, false, true);
            se.setPosition(canvasmargin, height - canvasmargin);
            se.render();
            svg.append(se.getSVG());
        }
        
        // Finish the SVG document
        svg.append(renderFooter());
        
        // Flush to disk
        try {
            File f = new File(entityName + ".svg");
            FileOutputStream o = new FileOutputStream(f);
            o.write(svg.toString().getBytes());
            o.flush();
            o.close();
            CLI.print("SVGRender: Wrote " + entityName + ".svg", 1);
        }
        catch (Exception ex) {
            ex.printStackTrace();
            System.exit(1);
        }
    }
    
    /** Renders a class diagram */
    public void renderClassDiagram(Diagram d) {
        
        handleDiagramLayout(d);

        // Render the layout as our body
        String body = layout.render();
        svg.append(renderHeader(layout.getWidth(), layout.getHeight()));
        svg.append(body);
        
        // Render links between linked elements
        renderDiagramLinks();
        
    }
    
    /** Renders a class diagram */
    public void renderUseCaseDiagram(Diagram d) {
        
        handleDiagramLayout(d);

        // Render the layout as our body
        String body = layout.render();
        svg.append(renderHeader(layout.getWidth(), layout.getHeight()));
        svg.append(body);
        
        // Render links between linked elements
        renderDiagramLinks();
        
    }
    
    /** 
     * Uses a diagram layout to find all the DiagramElement 
     * objects in it and render their links to other elements.
     */
    public void renderDiagramLinks() { 
        // Render the links between diagram elements
        // This works since all the SVGEntity objects held in the layout
        // know their positions now and we can accurately draw the links
        // between them.
        List o = layout.getObjects();
        CLI.print("SVGRenderer: Scanning links for " + o.size() + " visual objects.", 3);
        for (Iterator it = o.iterator(); it.hasNext(); ) {
            SVGEntity e1 = (SVGEntity) it.next();
            // For this entity, get its links to other entities
            if (e1.getEntity() instanceof DiagramElement) {
                CLI.print("SVGRenderer: Evaluating links for element '" + e1.getEntity().getName() + "'", 3);
                List links = ((DiagramElement) e1.getEntity()).getLinks();
                CLI.print("SVGRenderer: Element " + e1.getEntity().getName() + " has " + links.size() + " links.", 3);
                for (Iterator itt = links.iterator(); itt.hasNext(); ) {
                    DiagramLink l = (DiagramLink) itt.next();
                    // Get the target entity
                    DiagramElement te = l.getTargetEntity();
                    CLI.print("SVGRenderer: " + e1.getEntity().getName() + " -> " + te.getEntityName(), 3);
                    // Find the target entity's SVGEntity
                    for (Iterator ittt = o.iterator(); ittt.hasNext(); ) {
                        SVGEntity e2 = (SVGEntity) ittt.next();
                        if (e2.getEntity().getName().equals(te.getEntityName())) {
                            
                            // We've got the link and the other entity now, 
                            // render it.
                            svg.append(
                                new SVGLinkRenderer(e1, e2, layout.getObjects(), l.getLinkType()).renderLink()
                                );
                            break;
                        }
                    }
                }
            }
        }
    }
    /**
     * For any given diagram, creates the correct layout and 
     * adds all the diagram elements to it.
     * @param d
     */
    public void handleDiagramLayout(Diagram d) {
        
        // Create our layout manager.
        
        // Satellite layout
        if (d.getLayout().equals("satellite")) {
            layout = new SVGSatelliteLayout();
            SVGSatelliteLayout sat = (SVGSatelliteLayout) layout;
            Entity fe = null;
            if (d.getLayoutArgs().length == 0) {
                // No satellite class specified, take the first
                fe = (Entity) d.getElements().values().toArray()[0];
            }
            else {
                fe = (Entity) d.getElements().get(d.getLayoutArgs()[0]);
            }
            sat.setSatelliteEntity(wrapEntity(fe));
        }
        
        if (d.getLayout().equals("grid")) {
            int noCols = 1;
            try {
                noCols = Integer.parseInt(d.getLayoutArgs()[0]);
            }
            catch (NumberFormatException e) {
                CLI.print("SVGRenderer: (" + d.getName() + ") Grid layout arguments should start with number of columns.");
                System.exit(1);
            }
            layout = new SVGGridLayout(noCols, d.getLayoutArgs());
        }
        
        if (d.getLayout().equals("usecase")) {
            layout = new SVGUseCaseLayout();
        }
        
        // Loop through and add the diagram elements
        for (Iterator it = d.getElements().values().iterator(); it.hasNext(); ) {
            Entity e = (Entity) it.next();
            if (e != null) layout.add(wrapEntity(e));
        }
    }
    
    public SVGEntity wrapEntity(Entity e) {
        
        Entity etest = e;
        
        // If it's a diagram element, find out what entity is being
        // encapsulated
        if (e instanceof DiagramElement) 
            etest = ((DiagramElement) e).getEntity();
        
        // Check for different types of entity and wrap them
        if (etest instanceof Class)
            return new SVGClassRenderer(e);
        else if (etest instanceof Interface)
            return new SVGInterfaceRenderer(e);
        else if (etest instanceof Actor)
            return new SVGActorRenderer(e);
        else if (etest instanceof UseCase)
            return new SVGUseCaseRenderer(e);
        
        // Failed, can't wrap
        throw new IllegalArgumentException("Can't wrap entity " + e.getName() + ", class type is " + e.getClass().getName());
    }
    
    
    public void renderSingleEntity(Entity en) {
        SVGEntity e = wrapEntity(en);
        e.setPosition(0, 0);
        e.render();
        svg.append(renderHeader(e.getSize().width, e.getSize().height));
        svg.append(e.getSVG());
    }
    
}
