package net.sf.umlspeed.svg;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

/**
 * Manages the layout of SVG objects in a satellite layout around
 * one object.
 */
public class SVGSatelliteLayout extends SVGLayout {

    private SVGEntity satellite = null;
    private List objects = new ArrayList();
    private int canvaswidth = 0;
    private int canvasheight = 0;
    
    public void add(SVGEntity e) {
        // Don't do anything for the satellite entity
        if (e.getEntity().getName().equals(satellite.getEntity().getName())) 
                return;
        objects.add(e);
    }

    public void remove(SVGEntity e) {
        objects.remove(e);
    }
    
    public void setSatelliteEntity(SVGEntity e) {
        satellite = e;
        objects.add(satellite);
    }
    
    public int getWidth() {
        return canvaswidth;
    }
    
    public int getHeight() {
        return canvasheight;
    }

    public List getObjects() {
        return objects;
    }
    
    public String render() {
        
        StringBuffer svg = new StringBuffer();
        
        // We arrange the objects into two rows at the top and bottom
        // with the satellite entity in the middle. A quick pass through
        // first figures out the largest item we're dealing with
        int biggestwidth = 0;
        int biggestheight = 0;
        for (Iterator it = objects.iterator(); it.hasNext(); ) {
            SVGEntity e = (SVGEntity) it.next();
            e.setPosition(0, 0);
            if (e.getSize().width > biggestwidth)
                biggestwidth = e.getSize().width;
            if (e.getSize().height > biggestheight)
                biggestheight = e.getSize().height;
        }
        
        // Render the satellite object as well so we can get its size
        satellite.setPosition(0, 0);
        
        // The canvasmargin is the gap we allow between objects and the
        // edge of the canvas
        int canvasmargin = 20;
        
        // Margin is the gap (horizontal and vertical) between the nearest
        // diagram element. The fewer the elements, the larger the margin
        // should be for clarity.
        int margin = 10;
        if ( objects.size() < 3)
            margin = 100;
        else if ( objects.size() < 5)
            margin = 70;
        else if ( objects.size() < 10)
            margin = 50;
        else if ( objects.size() < 20)
            margin = 20;
        else
            margin = 10;
        
        // Figure out the top and bottom lines, along with the overall
        // canvas size.
        int topRowY = canvasmargin;
        int bottomRowY = canvasmargin + biggestheight + satellite.getSize().height + (margin * 2);
        int itemsPerRow = ((objects.size()-1) / 2);
        if (itemsPerRow == 0) itemsPerRow = 1;
        canvaswidth = (canvasmargin * 2) + ((biggestwidth + margin) * (itemsPerRow+1));
        canvasheight = (canvasmargin * 2) + (biggestheight * 2) + (margin * 3) + satellite.getSize().height; 
        
        // Render the satellite in its final position
        int satx = (canvaswidth / 2) - (satellite.getSize().width / 2);
        int saty = biggestheight + canvasmargin + margin;
        satellite.setPosition(satx, saty);
        svg.append(satellite.getSVG());
        
        // Render all the other objects
        int currentItem = 0;
        int y = topRowY;
        int x = canvasmargin;
        for (Iterator it = objects.iterator(); it.hasNext(); ) {
            
            SVGEntity e = (SVGEntity) it.next();
            
            // Ignore the satellite entity - we've already rendered it
            if (!e.getEntity().getName().equals(satellite.getEntity().getName())) {
                e.setPosition(centerInCell(new SVGDimension(biggestwidth, biggestheight), e.getSize(), new SVGPosition(x, y)));
                svg.append(e.getSVG());
                
                x += biggestwidth + margin;
                currentItem++;
                if (currentItem > itemsPerRow) {
                    y = bottomRowY;
                    x = canvasmargin;
                    currentItem = -1;
                }
            }
        }
        
        return svg.toString();
    }

}
