package net.sf.umlspeed.svg;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import net.sf.umlspeed.cli.CLI;
import net.sf.umlspeed.entities.Actor;
import net.sf.umlspeed.entities.DiagramElement;
import net.sf.umlspeed.entities.UseCase;

/**
 * Manages the layout of SVG objects in 2 vertical columns 
 * according to entity type. Actors on the left, use cases
 * on the right
 * 
 */
public class SVGUseCaseLayout extends SVGLayout {

    private List objects = new ArrayList();
    
    private int canvaswidth = 0;
    private int canvasheight = 0;
    
    public void add(SVGEntity e) {
        objects.add(e);
    }

    public void remove(SVGEntity e) {
        objects.remove(e);
    }

    public List getObjects() {
        return objects;
    }
    
    public int getWidth() {
        return canvaswidth;
    }
    
    public int getHeight() {
        return canvasheight;
    }
    
    public String render() {
        
        StringBuffer svg = new StringBuffer();
        
        // We arrange the objects into two rows at the top and bottom
        // with the satellite entity in the middle. A quick pass through
        // first figures out the largest item we're dealing with
        int biggestwidth = 0;
        int biggestheight = 0;
        for (Iterator it = objects.iterator(); it.hasNext(); ) {
            SVGEntity e = (SVGEntity) it.next();
            e.setPosition(0, 0);
            if (e.getSize().width > biggestwidth)
                biggestwidth = e.getSize().width;
            if (e.getSize().height > biggestheight)
                biggestheight = e.getSize().height;
        }
        
        // The canvasmargin is the gap we allow between objects and the
        // edge of the canvas
        int canvasmargin = 20;
        
        // Margin is the gap (horizontal and vertical) between the nearest
        // diagram element. The fewer the elements, the larger the margin
        // should be for clarity.
        int margin = 10;
        if ( objects.size() < 3)
            margin = 100;
        else if ( objects.size() < 5)
            margin = 70;
        else if ( objects.size() < 10)
            margin = 50;
        else if ( objects.size() < 20)
            margin = 20;
        else
            margin = 10;
        
        int leftcol = canvasmargin;
        int rightcol = canvasmargin + biggestwidth;
        int ly = canvasmargin;
        int ry = canvasmargin;
        
        // Render the objects
        int x = 0;
        int y = 0;
        for (Iterator it = objects.iterator(); it.hasNext(); ) {
            
            SVGEntity e = (SVGEntity) it.next();
            
            // If we're an Actor, go to the left column
            if (((DiagramElement) e.getEntity()).getEntity() instanceof Actor) {
                x = leftcol;
                y = ly;
                
                e.setPosition(centerInCell(new SVGDimension(biggestwidth, biggestheight), e.getSize(), new SVGPosition(x, y)));
                svg.append(e.getSVG());
                
                ly += margin + biggestheight;
            }
            else if (((DiagramElement) e.getEntity()).getEntity() instanceof UseCase) {
                x = rightcol;
                y = ry;
                
                e.setPosition(centerInCell(new SVGDimension(biggestwidth, biggestheight), e.getSize(), new SVGPosition(x, y)));
                svg.append(e.getSVG());
                
                ry += margin + biggestheight;
            }
            else {
                CLI.print("SVGUseCaseLayout: UseCase layout can only have Actor or UseCase entities.");
                System.exit(1);
            }
        }
        
        canvasheight = canvasmargin + ( ly > ry ? ly : ry);
        canvaswidth = (canvasmargin * 2) + ((biggestwidth + margin) * 2);
        
        
        return svg.toString();
    }

}
